"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _errors = require("../../auth/errors");
var _not_found = _interopRequireDefault(require("./../../backend/errors/not_found"));
/*
 *    Copyright 2020 floragunn GmbH
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * The SearchGuard  backend.
 */
class SearchGuardConfigurationBackend {
  constructor({
    elasticsearch
  }) {
    (0, _defineProperty2.default)(this, "_client", async ({
      headers = {},
      asWho = 'asCurrentUser',
      ...options
    }) => {
      const result = await this.elasticsearch.client.asScoped({
        headers
      })[asWho].transport.request(options);
      return result;
    });
    (0, _defineProperty2.default)(this, "restapiinfo", async headers => {
      try {
        return await this._client({
          path: '/_searchguard/api/permissionsinfo',
          method: 'get',
          headers: headers
        });
      } catch (error) {
        if (error.statusCode === 401) {
          throw new _errors.AuthenticationError(error.message, error);
        }
        throw error;
      }
    });
    (0, _defineProperty2.default)(this, "indices", async ({
      headers,
      index = []
    } = {}) => {
      try {
        const response = await this.elasticsearch.client.asScoped({
          headers
        }).asCurrentUser.cat.indices({
          index,
          format: 'json',
          h: 'index,health'
        });
        return response;
      } catch (error) {
        if (error.statusCode === 401) {
          throw new _errors.AuthenticationError(error.message, error);
        }
        throw error;
      }
    });
    (0, _defineProperty2.default)(this, "aliases", async ({
      headers,
      alias = []
    } = {}) => {
      try {
        const response = await this.elasticsearch.client.asScoped({
          headers
        }).asCurrentUser.cat.aliases({
          alias,
          format: 'json',
          h: 'index,alias'
        });
        return response;
      } catch (error) {
        if (error.statusCode === 401) {
          throw new _errors.AuthenticationError(error.message, error);
        }
        throw error;
      }
    });
    (0, _defineProperty2.default)(this, "dataStreams", async ({
      headers,
      index = []
    } = {}) => {
      try {
        const response = await this.elasticsearch.client.asScoped({
          headers
        }).asCurrentUser.indices.getDataStream({
          name: '*',
          format: 'json'
        });
        return response;
      } catch (error) {
        if (error.statusCode === 401) {
          throw new _errors.AuthenticationError(error.message, error);
        }
        throw error;
      }
    });
    (0, _defineProperty2.default)(this, "list", async (headers, resourceName) => {
      try {
        return await this._client({
          path: `/_searchguard/api/${resourceName}`,
          method: 'get',
          headers
        });
      } catch (error) {
        if (error.statusCode === 401) {
          throw new _errors.AuthenticationError(error.message, error);
        }
        throw error;
      }
    });
    (0, _defineProperty2.default)(this, "get", async (headers, resourceName, id) => {
      try {
        const response = await this._client({
          path: `/_searchguard/api/${resourceName}/${encodeURIComponent(id)}`,
          method: 'get',
          headers
        });
        return response[id];
      } catch (error) {
        if (error.statusCode === 401) {
          throw new _errors.AuthenticationError(error.message, error);
        }
        if (error.statusCode === 404) {
          throw new _not_found.default(error.message, error);
        }
        throw error;
      }
    });
    (0, _defineProperty2.default)(this, "save", async (headers, resourceName, id, body) => {
      try {
        return await this._client({
          path: `/_searchguard/api/${resourceName}/${encodeURIComponent(id)}`,
          method: 'put',
          headers,
          body
        });
      } catch (error) {
        if (error.statusCode === 401) {
          throw new _errors.AuthenticationError(error.message, error);
        }
        if (error.statusCode === 404) {
          throw new _not_found.default(error.message, error);
        }
        throw error;
      }
    });
    (0, _defineProperty2.default)(this, "delete", async (headers, resourceName, id) => {
      try {
        return await this._client({
          path: `/_searchguard/api/${resourceName}/${encodeURIComponent(id)}`,
          method: 'delete',
          headers
        });
      } catch (error) {
        if (error.statusCode === 401) {
          throw new _errors.AuthenticationError(error.message, error);
        }
        if (error.statusCode === 404) {
          throw new _not_found.default(error.message, error);
        }
        throw error;
      }
    });
    (0, _defineProperty2.default)(this, "clearCache", async headers => {
      try {
        return await this._client({
          path: '/_searchguard/api/cache',
          method: 'delete',
          headers
        });
      } catch (error) {
        if (error.statusCode === 401) {
          throw new _errors.AuthenticationError(error.message, error);
        }
        throw error;
      }
    });
    (0, _defineProperty2.default)(this, "validateDls", async (headers, indexname, body) => {
      try {
        return await this._client({
          path: '/_validate/query?explain=true',
          method: 'post',
          headers,
          body
        });
      } catch (error) {
        if (error.statusCode === 401) {
          throw new _errors.AuthenticationError(error.message, error);
        }
        throw error;
      }
    });
    (0, _defineProperty2.default)(this, "getIndexMappings", async ({
      headers,
      body: {
        index
      }
    }) => {
      try {
        const mappings = await this.elasticsearch.client.asScoped({
          headers
        }).asCurrentUser.indices.getMapping({
          index: index.join(','),
          ignore_unavailable: true,
          allow_no_indices: true
        });
        return {
          total: Object.keys(mappings).length,
          mappings
        };
      } catch (error) {
        if (error.statusCode === 401) {
          throw new _errors.AuthenticationError(error.message, error);
        }
        throw error;
      }
    });
    this.elasticsearch = elasticsearch;
  }
}
exports.default = SearchGuardConfigurationBackend;
module.exports = exports.default;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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